<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\UploadedFile;

class EmployeeProfile extends ActiveRecord
{
    const STATUS_NEW = 'Новая';
    const STATUS_IN_PROGRESS = 'Идет проверка данных';
    const STATUS_ACCEPTED = 'Данные приняты';

    public $photoFile;
    public $passportScanFile;
    public $diplomaScanFile;
    public $snilsScanFile;

    public static function tableName()
    {
        return '{{%employee_profile}}';
    }

    public function rules()
    {
        return [
            [['user_id'], 'required'],
            [['user_id', 'work_experience', 'created_at', 'updated_at'], 'integer'],
            
            [['gender', 'citizenship', 'marital_status', 'education', 'status'], 'string'],
            [['passport_issued_by', 'registration_address', 'health_status', 'additional_info'], 'string'],
            [['photo', 'diploma_scan', 'snils_scan'], 'string', 'max' => 500],
            [['educational_institution', 'specialization', 'work_place', 'position'], 'string', 'max' => 255],
            [['passport_series', 'passport_number', 'diploma_series_number', 'snils'], 'string', 'max' => 20],
            
            // Валидация файлов
            [['photoFile'], 'file', 'skipOnEmpty' => true, 'extensions' => 'jpg, jpeg, pdf', 'maxSize' => 10 * 1024 * 1024],
            [['passportScanFile', 'diplomaScanFile'], 'file', 'skipOnEmpty' => true, 'extensions' => 'jpg, jpeg, pdf', 'maxSize' => 20 * 1024 * 1024],
            [['snilsScanFile'], 'file', 'skipOnEmpty' => true, 'extensions' => 'jpg, jpeg, pdf', 'maxSize' => 10 * 1024 * 1024],
        ];
    }

    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'photo' => 'Фото сотрудника',
            'birth_date' => 'Дата рождения',
            'gender' => 'Пол',
            'citizenship' => 'Гражданство',
            'passport_series' => 'Серия паспорта',
            'passport_number' => 'Номер паспорта',
            'passport_issued_by' => 'Кем и когда выдан',
            'passport_issued_date' => 'Дата выдачи паспорта',
            'registration_address' => 'Адрес регистрации',
            'passport_scan' => 'Скан паспорта',
            'marital_status' => 'Семейное положение',
            'education' => 'Образование',
            'educational_institution' => 'Образовательная организация',
            'specialization' => 'Специальность',
            'diploma_series_number' => 'Серия и номер диплома',
            'diploma_scan' => 'Скан диплома',
            'snils' => 'СНИЛС',
            'snils_scan' => 'Скан СНИЛС',
            'work_place' => 'Место работы',
            'position' => 'Должность',
            'work_experience' => 'Стаж работы',
            'health_status' => 'Состояние здоровья',
            'additional_info' => 'Дополнительная информация',
            'status' => 'Статус',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->created_at = time();
                $this->status = self::STATUS_NEW;
            }
            $this->updated_at = time();
            
            // Обработка загрузки файлов
            $this->uploadFiles();
            
            return true;
        }
        return false;
    }

    public function uploadFiles()
    {
        $uploadPath = Yii::getAlias('@webroot/uploads/');
        if (!is_dir($uploadPath)) {
            mkdir($uploadPath, 0777, true);
        }

        $this->uploadFile('photoFile', 'photo', $uploadPath . 'photos/');
        $this->uploadFile('passportScanFile', 'passport_scan', $uploadPath . 'passports/');
        $this->uploadFile('diplomaScanFile', 'diploma_scan', $uploadPath . 'diplomas/');
        $this->uploadFile('snilsScanFile', 'snils_scan', $uploadPath . 'snils/');
    }

    private function uploadFile($fileAttribute, $dbAttribute, $path)
    {
        $file = UploadedFile::getInstance($this, $fileAttribute);
        if ($file) {
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = uniqid() . '_' . $file->baseName . '.' . $file->extension;
            if ($file->saveAs($path . $fileName)) {
                $this->$dbAttribute = $fileName;
            }
        }
    }

    // Relations
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    public static function getStatuses()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_IN_PROGRESS => 'Идет проверка данных',
            self::STATUS_ACCEPTED => 'Данные приняты',
        ];
    }
    
    /**
     * Преобразует дату из формата формы в формат базы данных
     */
    public function beforeValidate()
    {
        if (parent::beforeValidate()) {
            // Преобразуем даты из формата d.m.Y в Y-m-d если нужно
            if (!empty($this->birth_date) && strpos($this->birth_date, '.') !== false) {
                $date = \DateTime::createFromFormat('d.m.Y', $this->birth_date);
                if ($date) {
                    $this->birth_date = $date->format('Y-m-d');
                }
            }
            
            if (!empty($this->passport_issued_date) && strpos($this->passport_issued_date, '.') !== false) {
                $date = \DateTime::createFromFormat('d.m.Y', $this->passport_issued_date);
                if ($date) {
                    $this->passport_issued_date = $date->format('Y-m-d');
                }
            }
            
            return true;
        }
        return false;
    }
}